#!/usr/bin/env bash
set -euo pipefail

# -----------------------------------------------------------------------------
# 依赖检查：aria2c、tmux 和 ufw（如果存在）
# -----------------------------------------------------------------------------
if ! command -v aria2c &> /dev/null; then
  echo "[INFO] Installing aria2..."
  sudo apt-get update && sudo apt-get install -y aria2
fi

if ! command -v tmux &> /dev/null; then
  echo "[INFO] Installing tmux..."
  sudo apt-get update && sudo apt-get install -y tmux
fi

# （可选）如果你没有 ufw，也可以忽略此依赖检查
if command -v ufw &> /dev/null; then
  echo "[INFO] ufw detected."
else
  echo "[INFO] ufw not installed or not in PATH; skipping firewall rules."
fi

# -----------------------------------------------------------------------------
# 查找 .torrent 文件
# -----------------------------------------------------------------------------
shopt -s nullglob
torrent_files=( *.torrent )

if [[ ${#torrent_files[@]} -eq 0 ]]; then
  echo "[INFO] 当前目录下没有找到 .torrent 文件，退出。"
  exit 0
fi

# -----------------------------------------------------------------------------
# 针对每个 .torrent，创建 tmux 会话并启动做种（随机高端口 + ufw 放行）
# -----------------------------------------------------------------------------
for torrent in "${torrent_files[@]}"; do
  # 会话名：seed-<basename>
  base="$(basename "$torrent" .torrent)"
  session_name="seed-$base"

  # 如果会话已存在则跳过
  if tmux has-session -t "$session_name" 2> /dev/null; then
    echo "[INFO] 会话 $session_name 已存在，跳过。"
    continue
  fi

  # 随机选一个高端口（49152–65535）
  if command -v shuf &> /dev/null; then
    port=$(shuf -i 49152-65535 -n 1)
  else
    port=$(( RANDOM % 16384 + 49152 ))
  fi

  # 如果 ufw 已启用，则添加放行规则；若已放行则输出提示
  if command -v ufw &> /dev/null && sudo ufw status | grep -q "Status: active"; then
    echo "[INFO] 放行端口 $port/tcp 到 ufw..."
    sudo ufw allow "${port}/tcp" || echo "[INFO] 端口 $port 已经放行"
  fi

  echo "[INFO] 启动会话 $session_name，做种文件：$torrent （端口：$port）"
  tmux new-session -d -s "$session_name" -- \
    aria2c \
      --enable-dht=true \
      --enable-dht6=true \
      --enable-peer-exchange=true \
      --bt-seed-unverified=true \
      --bt-save-metadata=true \
      --dir="$(pwd)" \
      --listen-port="$port" \
      "$torrent" || echo "$session_name 启动失败。"
done

echo "[INFO] 所有做种会话已启动。"
