#!/usr/bin/env bash
set -euo pipefail

# 判断当前 APT 源格式状态
check_apt_format() {
    local old_format=false
    local new_format=false
    
    # 检查老格式 (.list)
    if [ -f "/etc/apt/sources.list" ]; then
        if grep -v '^#' /etc/apt/sources.list | grep -q '[^[:space:]]'; then
            old_format=true
        fi
    fi
    
    # 检查新格式中的 ubuntu.sources 文件
    if [ -f "/etc/apt/sources.list.d/ubuntu.sources" ]; then
        if grep -v '^#' /etc/apt/sources.list.d/ubuntu.sources | grep -q '[^[:space:]]'; then
            new_format=true
        fi
    fi
    
    # 返回状态
    if $old_format && $new_format; then
        echo "both"
    elif $old_format; then
        echo "old"
    elif $new_format; then
        echo "new"
    else
        echo "none"
    fi
}

# 寻找最快的镜像源
find_fastest_mirror() {
    # 所有要显示的内容都重定向到标准错误 >&2
    echo "正在测试镜像源速度..." >&2
    
    # 获取当前 Ubuntu 代号
    codename=$(lsb_release -cs)
    
    # 定义潜在的镜像源列表
    mirrors=(
        "https://archive.ubuntu.com/ubuntu/"
        "https://mirror.aarnet.edu.au/pub/ubuntu/archive/" # Australia
        "https://mirror.fsmg.org.nz/ubuntu/"               # New Zealand
        "https://mirrors.neterra.net/ubuntu/archive/"       # Bulgaria
        "https://mirror.csclub.uwaterloo.ca/ubuntu/"        # Canada
        "https://mirrors.dotsrc.org/ubuntu/"                # Denmark
        "https://mirrors.nic.funet.fi/ubuntu/"              # Finland
        "https://mirror.ubuntu.ikoula.com/"                 # France
        "https://mirror.xtom.com.hk/ubuntu/"                # Hong Kong
        "https://mirrors.piconets.webwerks.in/ubuntu-mirror/ubuntu/" # India
        "https://ftp.udx.icscoe.jp/Linux/ubuntu/"           # Japan
        "https://ftp.kaist.ac.kr/ubuntu/"                   # Korea
        "https://ubuntu.mirror.garr.it/ubuntu/"             # Italy
        "https://ftp.uni-stuttgart.de/ubuntu/"              # Germany
        "https://mirror.i3d.net/pub/ubuntu/"                # Netherlands
        "https://mirroronet.pl/pub/mirrors/ubuntu/"         # Poland
        "https://ubuntu.mobinhost.com/ubuntu/"              # Iran
        "http://sg.archive.ubuntu.com/ubuntu/"              # Singapore
        "http://ossmirror.mycloud.services/os/linux/ubuntu/" # Singapore
        "https://mirror.enzu.com/ubuntu/"                   # United States
        "http://jp.archive.ubuntu.com/ubuntu/"              # Japan
        "http://kr.archive.ubuntu.com/ubuntu/"              # Korea
        "http://us.archive.ubuntu.com/ubuntu/"              # United States
        "http://tw.archive.ubuntu.com/ubuntu/"              # Taiwan
        "https://mirror.twds.com.tw/ubuntu/"                # Taiwan
        "https://ubuntu.mirrors.uk2.net/ubuntu/"            # United Kingdom
        "http://mirrors.ustc.edu.cn/ubuntu/"                # 中国科学技术大学
        "http://ftp.sjtu.edu.cn/ubuntu/"                    # 上海交通大学
        "http://mirrors.tuna.tsinghua.edu.cn/ubuntu/"       # 清华大学
        "http://mirrors.aliyun.com/ubuntu/"                 # 阿里云
        "http://mirrors.163.com/ubuntu/"                    # 网易
        "http://mirrors.cloud.tencent.com/ubuntu/"          # 腾讯云
        "http://mirror.aiursoft.cn/ubuntu/"                 # Aiursoft
        "http://mirrors.huaweicloud.com/ubuntu/"            # 华为云
        "http://mirrors.zju.edu.cn/ubuntu/"                 # 浙江大学
        "http://azure.archive.ubuntu.com/ubuntu/"           # Azure
        "https://mirrors.isu.net.sa/apt-mirror/"            # Saudi Arabia
        "https://mirror.team-host.ru/ubuntu/"               # Russia
        "https://labs.eif.urjc.es/mirror/ubuntu/"           # Spain
        "https://mirror.alastyr.com/ubuntu/ubuntu-archive/" # Turkey
        "https://ftp.acc.umu.se/ubuntu/"                    # Sweden
        "https://mirror.kku.ac.th/ubuntu/"                  # Thailand
        "https://mirror.bizflycloud.vn/ubuntu/"             # Vietnam
    )
    
    declare -A results
    
    # 测试单个镜像源的速度
    for mirror in "${mirrors[@]}"; do
        echo "测试 $mirror ..." >&2
        response="$(curl -o /dev/null -s -w "%{http_code} %{time_total}\n" \
                  --connect-timeout 1 --max-time 3 "${mirror}dists/${codename}/Release")"
        
        http_code=$(echo "$response" | awk '{print $1}')
        time_total=$(echo "$response" | awk '{print $2}')
        
        if [ "$http_code" -eq 200 ]; then
            results["$mirror"]="$time_total"
            echo "  成功: $time_total 秒" >&2
        else
            echo "  失败: HTTP 代码 $http_code" >&2
            results["$mirror"]="9999"
        fi
    done
    
    # 按响应时间排序
    sorted_mirrors="$(
        for url in "${!results[@]}"; do
            echo "$url ${results[$url]}"
        done | sort -k2 -n
    )"
    
    echo >&2
    echo "=== 按响应时间排序的镜像源 (升序) ===" >&2
    echo "$sorted_mirrors" >&2
    echo >&2
    
    # 选取最快的镜像源
    fastest_mirror="$(echo "$sorted_mirrors" | head -n 1 | awk '{print $1}')"
    
    if [[ "$fastest_mirror" == "" || "${results[$fastest_mirror]}" == "9999" ]]; then
        echo "没有找到可用的镜像源，使用默认镜像源" >&2
        fastest_mirror="http://archive.ubuntu.com/ubuntu/"
    fi
    
    echo "找到最快的镜像源: $fastest_mirror" >&2
    echo >&2
    
    # 只有这一行会返回给调用者，不带 >&2
    echo "$fastest_mirror"
}

# 生成旧格式配置文件
generate_old_format() {
    local mirror="$1"
    local codename="$2"
    
    echo "生成旧格式配置文件 /etc/apt/sources.list"
    
    sudo tee /etc/apt/sources.list >/dev/null <<EOF
deb $mirror $codename main restricted universe multiverse
deb $mirror $codename-updates main restricted universe multiverse
deb $mirror $codename-backports main restricted universe multiverse
deb $mirror $codename-security main restricted universe multiverse
EOF
    
    echo "旧格式配置文件已更新"
}

# 生成新格式配置文件
generate_new_format() {
    local mirror="$1"
    local codename="$2"
    
    echo "生成新格式配置文件 /etc/apt/sources.list.d/ubuntu.sources"
    
    sudo tee /etc/apt/sources.list.d/ubuntu.sources >/dev/null <<EOF
Types: deb
URIs: $mirror
Suites: $codename
Components: main restricted universe multiverse
Signed-By: /usr/share/keyrings/ubuntu-archive-keyring.gpg

Types: deb
URIs: $mirror
Suites: $codename-updates
Components: main restricted universe multiverse
Signed-By: /usr/share/keyrings/ubuntu-archive-keyring.gpg

Types: deb
URIs: $mirror
Suites: $codename-backports
Components: main restricted universe multiverse
Signed-By: /usr/share/keyrings/ubuntu-archive-keyring.gpg

Types: deb
URIs: $mirror
Suites: $codename-security
Components: main restricted universe multiverse
Signed-By: /usr/share/keyrings/ubuntu-archive-keyring.gpg
EOF
    
    echo "新格式配置文件已更新"
}

# 主函数
main() {
    # 确保安装必要的包
    sudo apt update
    sudo apt install -y curl lsb-release
    
    # 获取当前格式
    format=$(check_apt_format)
    echo "当前 APT 源格式状态: $format"
    
    # 获取 Ubuntu 代号
    codename=$(lsb_release -cs)
    echo "Ubuntu 代号: $codename"
    
    # 获取最快的镜像源
    echo "正在寻找最快的镜像源..."
    fastest_mirror=$(find_fastest_mirror)
    
    # 根据格式决定更新策略
    case "$format" in
        "none")
            echo "未找到有效的 APT 源配置，将生成老格式配置文件"
            generate_old_format "$fastest_mirror" "$codename"
            ;;
        "old")
            echo "系统使用传统格式，将更新老格式配置文件"
            generate_old_format "$fastest_mirror" "$codename"
            ;;
        "new")
            echo "系统使用现代格式，将更新新格式配置文件"
            generate_new_format "$fastest_mirror" "$codename"
            ;;
        "both")
            echo "系统同时使用两种格式，将删除老格式，只保留新格式"
            sudo mv /etc/apt/sources.list /etc/apt/sources.list.bak
            echo "已备份旧格式文件到 /etc/apt/sources.list.bak"
            generate_new_format "$fastest_mirror" "$codename"
            ;;
    esac
    
    # 更新软件包列表
    echo "更新软件包列表..."
    sudo apt update
    
    echo "APT 源优化完成！"

    aptVersion=$(apt --version | head -n 1 | awk '{print $2}')
    echo "当前 APT 版本: $aptVersion"

    # If current APT version is 3.0 or higher, and using old format or none, convert to new format
    # sudo apt modernize-sources
    if [[ $(echo "$aptVersion >= 3.0" | bc) -eq 1 && ( "$format" == "old" || "$format" == "none" ) ]]; then
        echo "当前 APT 版本为 3.0 或更高，将转换为新格式"
        sudo apt modernize-sources
        echo "APT 源已转换为新格式"
    fi

}

# 执行主函数
main