#!/usr/bin/env bash
#===============================================================================
# Concise server preparation script with error confirmation
#===============================================================================

set -euo pipefail
export LC_ALL=C
export LANG=en_US.UTF-8
export DEBIAN_FRONTEND=noninteractive

#-----------------------------------
# Colors & Prompts
#-----------------------------------
Green="\033[32m"; Red="\033[31m"; Yellow="\033[33m"; Blue="\033[36m"; Font="\033[0m"
OK="${Green}[  OK  ]${Font}"; ERROR="${Red}[FAILED]${Font}"; WARN="${Yellow}[ WARN ]${Font}"

print_ok(){ echo -e "${OK} $1"; }
print_error(){ echo -e "${ERROR} $1"; }
print_warn(){ echo -e "${WARN} $1"; }

#-----------------------------------
# Error handling & confirmation
#-----------------------------------
on_error(){ print_error "Error at line $1."; areYouSure; }
trap 'on_error $LINENO' ERR

areYouSure(){
  print_warn "Continue despite errors? [y/N]"
  read -r ans
  case $ans in [yY]*) print_ok "Continuing...";; *) print_error "Aborted."; exit 1;; esac
}

#-----------------------------------
# Helpers
#-----------------------------------
run_local(){ print_ok "Local: $*"; "$@"; }
run_remote(){ sshpass -p "$REMOTE_PASS" ssh -o StrictHostKeyChecking=no "$REMOTE_USER@$SERVER" "$*"; }
wait_ssh(){
  print_ok "Waiting for SSH on $SERVER...(Running ssh $REMOTE_USER@$SERVER)"
  until sshpass -p "$REMOTE_PASS" ssh -q -o StrictHostKeyChecking=no "$REMOTE_USER@$SERVER" exit; do
    print_warn "SSH not ready, retrying in 5s..."
    sleep 5
  done
  print_ok "SSH available."
}

usage(){ echo "Usage: $0 <orig_user> <orig_pass> <server> <new_hostname> <new_user>"; exit 1; }

#-----------------------------------
# Main
#-----------------------------------
[ $# -ne 5 ] && usage
USER="$1"; PASS="$2"; SERVER="$3"; HOSTNAME="$4"; NEWUSER="$5"
REMOTE_USER="$USER"; REMOTE_PASS="$PASS"

# 1) Install sshpass locally
run_local sudo apt-get update -y
run_local sudo apt-get install -y sshpass

# 2) Clear known_hosts, wait for SSH
run_local ssh-keygen -R "$SERVER" -f ~/.ssh/known_hosts
wait_ssh

# 3) Hostname & reboot
print_ok "Setting hostname to $HOSTNAME"
run_remote "sudo hostnamectl set-hostname $HOSTNAME"
run_remote "sudo reboot" || true
print_ok "Server rebooting..."
sleep 5
wait_ssh

# 4) Create or verify new user
if run_remote "id -u $NEWUSER" &>/dev/null; then
  print_ok "User $NEWUSER exists"
else
  print_ok "Creating user $NEWUSER"
  run_remote "sudo adduser --disabled-password --gecos '' $NEWUSER"
fi

# 5) Grant sudo & set up passwordless
print_ok "Granting sudo to $NEWUSER"
run_remote "sudo usermod -aG sudo $NEWUSER"

print_ok "Setting passwordless sudo for $NEWUSER"
run_remote "echo '$NEWUSER ALL=(ALL) NOPASSWD:ALL' | sudo tee /etc/sudoers.d/$NEWUSER"

# 6) Generate & set random password
PASS_NEW=$(uuidgen)
print_ok "Setting password for $NEWUSER"
run_remote "echo '$NEWUSER:$PASS_NEW' | sudo chpasswd"
print_ok "New password for $NEWUSER: $PASS_NEW"

# 7) Copy SSH key
[ ! -f ~/.ssh/id_rsa.pub ] && run_local ssh-keygen -t rsa -N "" -f ~/.ssh/id_rsa
print_ok "Copying SSH key"
sshpass -p "$PASS_NEW" ssh-copy-id -i ~/.ssh/id_rsa.pub "$NEWUSER@$SERVER"

# Switch to new user for subsequent operations
print_ok "Switching to new user $NEWUSER instead of $REMOTE_USER"
REMOTE_USER="$NEWUSER"; REMOTE_PASS="$PASS_NEW"
wait_ssh

# 8) Harden SSH
print_ok "Hardening SSH settings"
run_remote "sudo sed -i 's/PermitRootLogin yes/PermitRootLogin no/; s/PasswordAuthentication yes/PasswordAuthentication no/; s/#PubkeyAuthentication yes/PubkeyAuthentication yes/' /etc/ssh/sshd_config"
run_remote "sudo systemctl restart sshd"

# 9) Remove other non-system users
print_ok "Removing other users"
others=$(run_remote \
  "awk -F: -v skip='$NEWUSER' '\$3>=1000 && \$1!=skip {print \$1}' /etc/passwd")

for u in $others; do
  print_warn "Deleting user $u"
  run_remote "sudo pkill -u $u || true; sudo deluser --remove-home $u"
done

# 10) Reset machine-id
print_ok "Resetting machine-id"
run_remote "sudo rm -f /etc/machine-id /var/lib/dbus/machine-id"
run_remote "sudo systemd-machine-id-setup; sudo cp /etc/machine-id /var/lib/dbus/machine-id"

# 11) Enable UFW & OpenSSH
print_ok "Enabling UFW firewall"
run_remote "sudo apt-get install -y ufw"
run_remote "sudo ufw allow OpenSSH && echo y | sudo ufw enable"

# 12) Install & configure Fail2Ban
print_ok "Installing Fail2Ban"
run_remote "sudo apt-get update"
run_remote "sudo apt-get install -y fail2ban"
print_ok "Configuring Fail2Ban"
run_remote <<'EOF'
sudo tee /etc/fail2ban/jail.local > /dev/null <<EOJ
[sshd]
enabled   = true
port      = ssh
filter    = sshd
logpath   = /var/log/auth.log
maxretry  = 3
findtime  = 600
bantime   = 3600
EOJ
sudo systemctl restart fail2ban
EOF
print_ok "Fail2Ban setup complete"
run_remote "sudo fail2ban-client status sshd"

# 13) Enable BBR
print_ok "Enabling BBR congestion control"
#run_remote "sudo bash -c 'grep -q bbr /etc/sysctl.conf || { echo net.core.default_qdisc=fq >>/etc/sysctl.conf; echo net.ipv4.tcp_congestion_control=bbr >>/etc/sysctl.conf; sysctl -p; }'"
run_remote <<'EOF'
sudo tee /etc/sysctl.d/99-bbr.conf > /dev/null <<SYSCTL
net.core.default_qdisc = fq
net.ipv4.tcp_congestion_control = bbr
SYSCTL
sudo sysctl --system
EOF

# 14) Select best mirror & update
print_ok "Selecting best mirror & updating"
run_remote "curl -s https://gist.aiursoft.cn/anduin/879917820a6c4b268fc12c21f1b3fe7a/raw/HEAD/mirror.sh | bash"
run_remote "sudo apt-get update"

# 15) Install latest HWE kernel
print_ok "Installing latest HWE kernel"
run_remote "sudo apt-get install -y \$(apt search linux-generic-hwe- | awk -F/ '/linux-generic-hwe-/{print \$1}' | head -1)"

# 16) Reboot & wait
print_ok "Rebooting server"
run_remote "sudo reboot" || true
sleep 5
wait_ssh

# 17) Final updates & cleanup
print_ok "Installing upgrades & cleanup"
run_remote "sudo apt-get update && sudo apt-get upgrade -y && sudo apt-get autoremove -y"

# 18) Performance tuning
print_ok "Tuning CPU performance & timezone"
run_remote "sudo apt-get install -y linux-tools-$(uname -r)"
run_remote "sudo apt-get install -y cpupower" || true
run_remote "sudo cpupower frequency-set -g performance" || true
run_remote "sudo timedatectl set-timezone GMT"

# 19) Remove snap
print_ok "Removing snapd"
run_remote "sudo systemctl disable --now snapd && sudo apt-get purge -y snapd && sudo rm -rf /snap /var/lib/snapd /var/cache/snapd"
run_remote "sudo tee /etc/apt/preferences.d/no-snap.pref <<EOF
Package: snapd
Pin: release a=*
Pin-Priority: -10
EOF"

# 20) Final cleanup & benchmark
print_ok "Final autoremove & benchmark"
run_remote "sudo apt-get autoremove -y --purge"
run_remote "sudo apt-get install -y sysbench && sysbench cpu --threads=\$(nproc) run"
run_remote "sudo apt-get autoremove -y sysbench --purge"
print_ok "Setup complete. Connect via: ssh $NEWUSER@$SERVER"

