#!/usr/bin/env python3
"""
为个人所有 GitLab 仓库批量添加 CI/CD 变量：
  DOCKER_USERNAME
  DOCKER_PASSWORD
  LOCAL_DOCKER_USERNAME
  LOCAL_DOCKER_PASSWORD
"""

import os
import sys
import requests

def get_user_id(base_url, headers, username):
    resp = requests.get(
        f"{base_url}/api/v4/users",
        params={"username": username},
        headers=headers
    )
    resp.raise_for_status()
    users = resp.json()
    if not users:
        print(f"用户 '{username}' 未找到", file=sys.stderr)
        sys.exit(1)
    return users[0]["id"]

def get_projects(base_url, headers, user_id):
    projects = []
    page = 1
    while True:
        resp = requests.get(
            f"{base_url}/api/v4/users/{user_id}/projects",
            params={"per_page": 100, "page": page},
            headers=headers
        )
        resp.raise_for_status()
        data = resp.json()
        if not data:
            break
        projects.extend(data)
        page += 1
    return projects

def delete_variable(base_url, headers, project_id, key):
    """无脑删除项目下的指定变量，忽略不存在的错误"""
    url = f"{base_url}/api/v4/projects/{project_id}/variables/{key}"
    r = requests.delete(url, headers=headers)
    # 204：删除成功；404：变量本就不存在，都视作 OK
    return r.status_code in (204, 404)

def add_variable(base_url, headers, project_id, key, value):
    payload = {
        "key": key,
        "value": value,
        "variable_type": "env_var",
        "protected": "true",            # 仅在受保护的分支/标签上可用
        "masked": "true",               # 在 Job 日志中掩码
        "masked_and_hidden": "true",    # 掩码并在 UI 中隐藏（不能再查看）
        "raw": "true",                  # 原样，不展开变量引用
        "environment_scope": "*"        # 生效所有环境
    }
    resp = requests.post(
        f"{base_url}/api/v4/projects/{project_id}/variables",
        headers=headers,
        data=payload
    )
    if resp.status_code == 201:
        return True, None
    elif resp.status_code == 400:
        return False, resp.text
    else:
        resp.raise_for_status()

def main():
    base_url = os.getenv("GITLAB_BASE_URL", "https://gitlab.aiursoft.cn")
    token = "glpat-9ztZJRPATvsuaQ59yBxZ"
    headers = {"Private-Token": token}

    username = os.getenv("GITLAB_USERNAME", "anduin")

    user_id = get_user_id(base_url, headers, username)
    projects = get_projects(base_url, headers, user_id)

    variables = [
        ("DOCKER_USERNAME", "aaaaa"),
        ("DOCKER_PASSWORD", "bbbbb"),
        ("LOCAL_DOCKER_USERNAME", "ccccc"),
        ("LOCAL_DOCKER_PASSWORD", "ddddd"),
        ("LOCAL_NUGET_API_KEY", "eeeeeeee"),
        ("NUGET_API_KEY", "fffff"),
    ]

    for proj in projects:
        pid = proj["id"]
        name = proj.get("path_with_namespace", proj["name"])
        print(f"▶ 处理项目：{name}")

        # —— 新增：先无脑删除所有旧变量 —— 
        for key, _ in variables:
            deleted = delete_variable(base_url, headers, pid, key)
            status = "已删除" if deleted else "删除失败"
            print(f"    ↳ 删除 {key}: {status}")

        # —— 再批量添加 —— 
        for key, val in variables:
            ok, msg = add_variable(base_url, headers, pid, key, val)
            if ok:
                print(f"    ✔ 添加 {key}")
            else:
                print(f"    ✖ 添加 {key} 失败：{msg}")

if __name__ == "__main__":
    main()
