#!/bin/bash

# 辅助函数，功能等同于 PowerShell 的 Clone-GitRepositories
# $1: 包含仓库对象的 JSON 数组字符串
# $2: 目标基础路径
clone_git_repositories() {
    local repos_json="$1"
    local destination_path="$2"

    # 使用 jq -c '.[]' 来迭代 JSON 数组中的每个对象
    # 'while read -r repo' 逐行读取每个对象
    echo "$repos_json" | jq -c '.[]' | while read -r repo; do
        # 跳过被 archive 的仓库
        local is_archived
        is_archived=$(echo "$repo" | jq -r '.archived')
        if [[ "$is_archived" == "true" ]]; then
            local repo_name
            repo_name=$(echo "$repo" | jq -r '.name')
            echo "Skipping archived repository: $repo_name"
            continue
        fi

        local repo_name
        repo_name=$(echo "$repo" | jq -r '.name')
        local repo_url
        repo_url=$(echo "$repo" | jq -r '.ssh_url_to_repo')

        # 判断 topics 是否存在且有内容，取出第一个 topic；否则使用 "no-topic"
        local first_topic
        local topics_count
        topics_count=$(echo "$repo" | jq -r '.topics | length')

        if [[ "$topics_count" -gt 0 ]]; then
            first_topic=$(echo "$repo" | jq -r '.topics[0]')
        else
            first_topic="no-topic"
        fi

        # 构造最终的 clone 目标路径：$destination_path/$first_topic/$repo_name
        local topic_path="$destination_path/$first_topic"
        
        # 'mkdir -p' 会创建所有必需的父目录，且如果目录已存在也不会报错
        # 这替代了 Test-Path 和 New-Item
        mkdir -p "$topic_path"
        
        local repo_path="$topic_path/$repo_name"

        if [[ ! -d "$repo_path" ]]; then
            git clone "$repo_url" "$repo_path"
            echo "Cloned $repo_name to $repo_path"
        else
            echo "$repo_name already exists at $repo_path, skipping."
        fi
    done
}

# 主函数，功能等同于 PowerShell 的 Reset-GitRepos
reset_git_repos() {
    echo "Deleting items..."
    # Bash 使用正斜杠 '/' 作为路径分隔符
    # 'rm -rf' 强制递归删除，且在目录不存在时不报错，类似 -ErrorAction SilentlyContinue
    rm -rf "$HOME/Source/Repos/Aiursoft"
    rm -rf "$HOME/Source/Repos/Anduin"
    echo "Items deleted!"

    sleep 5

    # Bash 中使用 echo -e 和 ANSI 转义序列来输出颜色
    echo -e "\033[0;32mCloning all repos...\033[0m"
    local gitlab_base_url="https://gitlab.aiursoft.cn"
    local api_url="$gitlab_base_url/api/v4"
    local group_name="Aiursoft"
    local user_name="Anduin"

    local destination_path_aiursoft="$HOME/Source/Repos/Aiursoft"
    local destination_path_anduin="$HOME/Source/Repos/Anduin"

    # 'mkdir -p' 替代 Test-Path 和 New-Item
    mkdir -p "$destination_path_aiursoft"
    mkdir -p "$destination_path_anduin"

    # 使用 curl -s (silent) 获取数据，并通过 jq 解析
    local group_url="$api_url/groups?search=$group_name"
    local group_request
    group_request=$(curl -s "$group_url")
    local group_id
    group_id=$(echo "$group_request" | jq -r '.[0].id')

    local user_url="$api_url/users?username=$user_name"
    local user_request
    user_request=$(curl -s "$user_url")
    local user_id
    user_id=$(echo "$user_request" | jq -r '.[0].id')

    local repo_url_aiursoft="$api_url/groups/$group_id/projects?simple=true&per_page=999&visibility=public&page=1"
    local repo_url_anduin="$api_url/users/$user_id/projects?simple=true&per_page=999&visibility=public&page=1"

    # 使用 curl 获取数据，并立即用 jq 过滤掉 archived 的仓库
    # jq 'map(select(.archived != true))' 会过滤数组并返回一个新数组
    # 注意：PowerShell 中的 Where-Object { $_.archived -ne $true } 在这里被 jq 的 'select' 实现了
    local repos_aiursoft
    repos_aiursoft=$(curl -s "$repo_url_aiursoft" | jq '[.[] | select(.archived != true)]')
    local repos_anduin
    repos_anduin=$(curl -s "$repo_url_anduin" | jq '[.[] | select(.archived != true)]')

    # 调用辅助函数，传入 JSON 字符串和目标路径
    clone_git_repositories "$repos_aiursoft" "$destination_path_aiursoft"
    clone_git_repositories "$repos_anduin" "$destination_path_anduin"
}

# ---
# 如果你想让这个脚本被执行时自动运行，请取消下面这行的注释：
# reset_git_repos
#
# 如果你想像 'source' 一样导入这些函数，请保持原样。
# ---