#!/bin/bash

# daily_build_v2.sh
# Script Purpose:
# This script is scheduled to run daily at 21:00 UTC to:
# 1) For each branch (e.g. 1.1, 1.2):
#    a) Switch to the branch, ensure local code is up-to-date (clean -fdx, reset --hard, pull)
#    b) Check if there are new commits in the past 24 hours
#       - If yes, a build is required
#       - If no but the last build was more than a week ago, a forced build is required
#       - Otherwise, skip
#    c) If a build is required, run build_all_langs.sh in a specific tmux session,
#       then copy build artifacts from ./src/dists/* to ~/built/<branch>/
#
# Additional requirements for this version:
# - Do NOT build 1.1 and 1.2 at the same time. Must wait for 1.1 to finish before building 1.2.
# - If no build is needed, do NOT remove any existing build artifacts.
# - Provide more logs rather than comments.

REPO_DIR="$HOME/anduinos"
SRC_DIR="$REPO_DIR/src"
BUILD_SCRIPT="$SRC_DIR/build_all_langs.sh"
SESSION_NAME_PREFIX="build_session"
BASE_BUILT_DIR="$HOME/built"

# Branch list (example: 1.1 & 1.2)
BRANCHES=("1.1" "1.2")

# Print an error message and exit
error_exit() {
    echo "[$(date)] ERROR: $1" >&2
    exit 1
}

# Function to check and build a specific branch
check_and_build_branch() {
    local branch_name=$1
    
    local sanitized_branch_name="${branch_name//./_}"
    local session_name="${SESSION_NAME_PREFIX}_${sanitized_branch_name}"
    
    echo "[$(date)] INFO: Start checking branch: $branch_name"
    echo "[$(date)] INFO: The tmux session name will be: $session_name"

    local last_build_file="$HOME/.last_build_date_${branch_name}"

    # Switch to the repository directory
    echo "[$(date)] INFO: Changing directory to $REPO_DIR"
    cd "$REPO_DIR" || error_exit "Cannot change directory to $REPO_DIR"

    echo "[$(date)] INFO: Reset and fetch latest code from remote"
    git reset --hard HEAD
    git fetch

    echo "[$(date)] INFO: Switching to branch $branch_name"
    git switch "$branch_name" || error_exit "Failed to switch to branch $branch_name"

    echo "[$(date)] INFO: Pulling latest code"
    git pull || error_exit "Failed to pull latest code on branch $branch_name"

    # Check if new commit exists in the past 24 hours
    local BUILD_REQUIRED=false
    if git log -1 --since="24 hours ago" "$branch_name" > /dev/null 2>&1; then
        BUILD_REQUIRED=true
        echo "[$(date)] INFO: Found new commits in the past 24 hours for branch $branch_name"
    else
        # Check last build time
        if [ -f "$last_build_file" ]; then
            local last_build_date
            last_build_date=$(cat "$last_build_file")
            local last_build_epoch
            last_build_epoch=$(date -d "$last_build_date" +%s)
            local current_epoch
            current_epoch=$(date +%s)
            local seven_days_ago=$(( current_epoch - 7*86400 ))

            if [ "$last_build_epoch" -lt "$seven_days_ago" ]; then
                BUILD_REQUIRED=true
                echo "[$(date)] INFO: No recent commits, but last build on $last_build_date is older than 7 days; forcing a build."
            else
                echo "[$(date)] INFO: No new commits for branch $branch_name, and last build is within a week. Skipping build."
            fi
        else
            # If there's no record file, we consider it as needing a build
            BUILD_REQUIRED=true
            echo "[$(date)] INFO: No last_build_date record found. Forcing a build."
        fi
    fi

    # If build is required, proceed
    if [ "$BUILD_REQUIRED" = true ]; then
        echo "[$(date)] INFO: Build is required for branch $branch_name"

        # Optional: Clean up the old artifacts only when we are about to build
        echo "[$(date)] INFO: Removing old artifacts under $BASE_BUILT_DIR/$branch_name"
        rm -rf "$BASE_BUILT_DIR/$branch_name"

        # Ensure the build output directory exists
        mkdir -p "$BASE_BUILT_DIR/$branch_name"

        # Kill any old tmux session for this branch
        local session_name="${SESSION_NAME_PREFIX}_${branch_name}"
        echo "[$(date)] INFO: Checking and killing any existing tmux session named $session_name"
        if tmux has-session -t "$session_name" 2>/dev/null; then
            tmux kill-session -t "$session_name"
            echo "[$(date)] INFO: Killed existing tmux session: $session_name"
        fi

        # Start a new tmux session for build
        echo "[$(date)] INFO: Creating a new tmux session for building branch $branch_name"
        tmux new-session -d -s "$session_name" "cd '$SRC_DIR' && bash '$BUILD_SCRIPT'"

        if [ $? -eq 0 ]; then
            echo "[$(date)] INFO: Build script launched in tmux session: $session_name"

            # Wait until the build script finishes inside tmux
            # Option 1: Polling approach until session not found
            echo "[$(date)] INFO: Waiting for the tmux session ($session_name) to finish..."
            while tmux has-session -t "$session_name" 2>/dev/null; do
                sleep 5
            done

            echo "[$(date)] INFO: Tmux session ($session_name) finished building"

            # After build completes, copy artifacts
            echo "[$(date)] INFO: Copying build artifacts from $SRC_DIR/dists to $BASE_BUILT_DIR/$branch_name"
            cp -rf "$SRC_DIR/dists/"* "$BASE_BUILT_DIR/$branch_name"/

            # Update last build date
            date +"%Y-%m-%d" > "$last_build_file"
            echo "[$(date)] INFO: Updated last build date record for branch $branch_name"
        else
            error_exit "[$(date)] ERROR: Failed to start build script in tmux session for branch $branch_name"
        fi
    else
        echo "[$(date)] INFO: No build needed for branch $branch_name. Keeping existing artifacts."
    fi
}

# MAIN FLOW
# Process each branch in sequence. We do NOT parallel-build branches.
for branch in "${BRANCHES[@]}"; do
    check_and_build_branch "$branch"
done

echo "[$(date)] INFO: All branches processed. Script completed."
