#!/bin/bash

# daily_build_v2.sh (refactored without tmux)
# Script Purpose:
# 1) For each branch in the list:
#    a) Switch to the branch, ensure local code is up-to-date (clean -fdx, reset --hard, pull)
#    b) Check if there are new commits in the past 24 hours
#       - If yes, a build is required
#       - If no but the last build was more than a week ago, a forced build is required
#       - Otherwise, skip
#    c) If a build is required, run build_all_langs.sh in the current session,
#       then copy build artifacts from ./src/dists/* to ~/built/<branch>/
#
# Additional requirements:
# - Do NOT build branches in parallel. Must finish one before moving to the next.
# - If no build is needed, do NOT remove any existing build artifacts.
# - Provide more logs rather than comments.

REPO_DIR="/mnt/second/anduinos"
SRC_DIR="${REPO_DIR}/src"
BUILD_SCRIPT="${REPO_DIR}/build_all.sh"
BASE_BUILT_DIR="/mnt/second/built"
BUILD_MIRROR="https://ftp.uni-stuttgart.de/ubuntu/"

# Branch list (example: 1.1 & 1.3)
BRANCHES=("1.1" "1.3")

# Print an error message and exit
error_exit() {
    echo "[$(date)] ERROR: $1" >&2
    # Infinite sleep to keep the container running for debugging
    sleep 99999999
}

# Function to check and build a specific branch
check_and_build_branch() {
    local branch_name="$1"
    echo "[$(date)] INFO: Start checking branch: $branch_name"

    local last_build_file="$HOME/.last_build_date_${branch_name}"

    echo "[$(date)] INFO: Changing directory to $REPO_DIR"
    cd "$REPO_DIR" || error_exit "Cannot change directory to $REPO_DIR"

    echo "[$(date)] INFO: Reset and fetch latest code from remote"
    git reset --hard HEAD
    git fetch

    echo "[$(date)] INFO: Switching to branch $branch_name"
    git switch "$branch_name" || error_exit "Failed to switch to branch $branch_name"

    echo "[$(date)] INFO: Pulling latest code with timeout and retries"
    local RETRY_COUNT=0
    local MAX_RETRIES=10
    local PULL_SUCCESS=false

    while [ $RETRY_COUNT -lt $MAX_RETRIES ]; do
        echo "[$(date)] INFO: Pull attempt $((RETRY_COUNT + 1)) of $MAX_RETRIES"
        if timeout 300s git pull; then
            PULL_SUCCESS=true
            echo "[$(date)] INFO: git pull succeeded"
            break
        else
            echo "[$(date)] WARN: git pull attempt $((RETRY_COUNT + 1)) failed, retrying in 5s..."
            RETRY_COUNT=$((RETRY_COUNT + 1))
            sleep 5
        fi
    done

    if [ "$PULL_SUCCESS" != true ]; then
        error_exit "Failed to pull latest code on branch $branch_name after $MAX_RETRIES attempts"
    fi

    # Check if new commit exists in the past 24 hours
    local BUILD_REQUIRED=false
    if git log -1 --since="24 hours ago" "$branch_name" > /dev/null 2>&1; then
        BUILD_REQUIRED=true
        echo "[$(date)] INFO: Found new commits in the past 24 hours for branch $branch_name"
    else
        if [ -f "$last_build_file" ]; then
            local last_build_date
            last_build_date=$(cat "$last_build_file")
            local last_build_epoch
            last_build_epoch=$(date -d "$last_build_date" +%s)
            local current_epoch
            current_epoch=$(date +%s)
            local seven_days_ago=$(( current_epoch - 7*86400 ))
            if [ "$last_build_epoch" -lt "$seven_days_ago" ]; then
                BUILD_REQUIRED=true
                echo "[$(date)] INFO: No recent commits, but last build on $last_build_date is older than 7 days; forcing a build."
            else
                echo "[$(date)] INFO: No new commits for branch $branch_name, and last build is within a week. Skipping build."
            fi
        else
            BUILD_REQUIRED=true
            echo "[$(date)] INFO: No last_build_date record found. Forcing a build."
        fi
    fi

    if [ "$BUILD_REQUIRED" = true ]; then
        echo "[$(date)] INFO: Build is required for branch $branch_name"
        sed -i "s|export BUILD_UBUNTU_MIRROR=.*|export BUILD_UBUNTU_MIRROR=\"$BUILD_MIRROR\"|" "$SRC_DIR/args.sh"
        echo "[$(date)] INFO: Updated BUILD_UBUNTU_MIRROR in $SRC_DIR/args.sh"

        (cd "$REPO_DIR" && bash "$BUILD_SCRIPT" -c ./config/all.json)
        if [ $? -eq 0 ]; then
            echo "[$(date)] INFO: Build script completed successfully"
            echo "[$(date)] INFO: Removing old artifacts under $BASE_BUILT_DIR/$branch_name"
            rm -rf "$BASE_BUILT_DIR/$branch_name"
            mkdir -p "$BASE_BUILT_DIR/$branch_name"
            echo "[$(date)] INFO: Moving build artifacts from $SRC_DIR/dist to $BASE_BUILT_DIR/$branch_name"
            mv "$SRC_DIR/dist/"* "$BASE_BUILT_DIR/$branch_name"/
            date +"%Y-%m-%d" > "$last_build_file"
            echo "[$(date)] INFO: Updated last build date record for branch $branch_name"
        else
            error_exit "Build script failed for branch $branch_name"
        fi
    else
        echo "[$(date)] INFO: No build needed for branch $branch_name. Keeping existing artifacts."
    fi
}

# MAIN FLOW
for branch in "${BRANCHES[@]}"; do
    check_and_build_branch "$branch"
done

echo "[$(date)] INFO: All branches processed. Script completed."
