#!/bin/bash
set -e

# Help function
usage() {
    echo "Usage: sudo $0 <path-to-image>"
    echo "Changes the GDM background to the specified image."
    echo "Note: This script must be run as root."
    exit 1
}

# Check for root
if [ "$EUID" -ne 0 ]; then
    echo "Error: This script must be run as root."
    exit 1
fi

# Check arguments
TARGET_IMAGE="$1"
if [ -z "$TARGET_IMAGE" ]; then
    usage
fi

if [ ! -f "$TARGET_IMAGE" ]; then
    echo "Error: Image file '$TARGET_IMAGE' not found."
    exit 1
fi

# Check dependencies
if ! command -v glib-compile-resources &> /dev/null; then
    echo "Error: 'glib-compile-resources' is not installed."
    echo "Please install 'libglib2.0-dev-bin' (Ubuntu/Debian) or 'glib2-devel' (Fedora)."
    exit 1
fi

# Define Paths
REPO_DIR="$(dirname "$(readlink -m "${0}")")"
THEME_ASSETS_DIR="${REPO_DIR}/other/gdm/theme"
XML_FILE="${REPO_DIR}/other/gdm/gnome-shell-theme.gresource.xml"
WORK_DIR="/tmp/gdm-switch-wallpaper-work"

# Targeted GDM theme files (Priority order based on lib-core.sh)
POP_OS_GR_FILE="/usr/share/gnome-shell/theme/Pop/gnome-shell-theme.gresource"
YARU_GR_FILE="/usr/share/gnome-shell/theme/Yaru/gnome-shell-theme.gresource"
ZORIN_GR_DARK_FILE="/usr/share/gnome-shell/theme/ZorinBlue-Dark/gnome-shell-theme.gresource"
ZORIN_GR_LIGHT_FILE="/usr/share/gnome-shell/theme/ZorinBlue-Light/gnome-shell-theme.gresource"
MISC_GR_FILE="/usr/share/gnome-shell/gnome-shell-theme.gresource"

# Find which theme file is in use
TARGET_FILE=""
if [ -f "$POP_OS_GR_FILE" ]; then
    TARGET_FILE="$POP_OS_GR_FILE"
elif [ -f "$YARU_GR_FILE" ]; then
    TARGET_FILE="$YARU_GR_FILE"
elif [ -f "$ZORIN_GR_DARK_FILE" ]; then
    TARGET_FILE="$ZORIN_GR_DARK_FILE"
elif [ -f "$ZORIN_GR_LIGHT_FILE" ]; then
    TARGET_FILE="$ZORIN_GR_LIGHT_FILE"
elif [ -f "$MISC_GR_FILE" ]; then
    TARGET_FILE="$MISC_GR_FILE"
else
    echo "Error: Could not find any supported GDM theme .gresource file."
    exit 1
fi

echo "Target GDM theme file: $TARGET_FILE"

# Prepare Workspace
echo "Preparing temporary workspace at $WORK_DIR..."
rm -rf "$WORK_DIR"
mkdir -p "$WORK_DIR"

# Copy base theme assets
if [ ! -d "$THEME_ASSETS_DIR" ]; then
    echo "Error: Theme assets not found at $THEME_ASSETS_DIR"
    exit 1
fi
cp -r "$THEME_ASSETS_DIR"/* "$WORK_DIR/"

# Copy and rename the new background image
echo "Processing background image..."
cp "$TARGET_IMAGE" "$WORK_DIR/background.png"

# Backup existing theme file
if [ ! -f "${TARGET_FILE}.bak" ]; then
    echo "Backing up original theme to ${TARGET_FILE}.bak..."
    cp "$TARGET_FILE" "${TARGET_FILE}.bak"
else
    echo "Backup already exists at ${TARGET_FILE}.bak, skipping backup."
fi

# Compile Resources
echo "Compiling GDM resources..."
glib-compile-resources \
    --sourcedir="$WORK_DIR" \
    --target="$TARGET_FILE" \
    "$XML_FILE"

# Cleanup
rm -rf "$WORK_DIR"

echo "Success! GDM background has been updated."
echo "You may need to restart GDM or reboot to see the changes."
echo "  - To restart GDM: sudo systemctl restart gdm"
